using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>03/12/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// Address data access layer class.
	///</summary>

	#endregion
	
	public class BloodUnitModification
	{

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2118"> 
		///		<ExpectedInput>Product modification code</ExpectedInput>
		///		<ExpectedOutput>Data table containing blood units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2119"> 
		///		<ExpectedInput>Invalid ModificationCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function to search database for units that can be modified
		/// </summary>
		/// <param name="productModification">Product modification type</param>
		/// <param name="modificationMethod">Open,Closed, SCD</param>
		/// <returns>DataTable containing blood units data</returns>
		/// <param name="includeAllStatus">Not filter out invalid stati</param>
		public static DataTable GetModifiableBloodUnits(Common.ProductModification productModification, Common.ModifyUnitMethod modificationMethod, bool includeAllStatus)
		{
			SqlParameter[] _prms =
			{
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetModifiableBloodUnits.divisioncode, Common.LogonUser.LogonUserDivisionCode),
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetModifiableBloodUnits.productmodificationcode, Common.Utility.GetProductModificationCharFromEnum(productModification)),
				new SqlParameter(STOREDPROC.GetModifiableBloodUnits.modificationmethod, SqlDbType.Int),
				new SqlParameter(STOREDPROC.GetModifiableBloodUnits.includeallstatus, SqlDbType.Bit)
			};
            			
			_prms[2].Value = (modificationMethod == Common.ModifyUnitMethod.Open) ? 1 : 0;

			return Common.StoredProcedure.GetData(STOREDPROC.GetModifiableBloodUnits.StoredProcName, _prms).Tables[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7497"> 
		///		<ExpectedInput>Parameters for Id selection</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7498"> 
		///		<ExpectedInput>Invlaid Parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the first Unused Pool Id
		/// </summary>
		/// <param name="counterYear">2-digit string representation of year</param>
		/// <returns></returns>
		public static System.Data.DataTable GetPoolCounter(string counterYear)
		{
            //CR3320 - removed parameters after sproc was streamlined
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPoolCounter.counteryear, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetPoolCounter.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = counterYear;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return(Common.StoredProcedure.GetData(STOREDPROC.GetPoolCounter.StoredProcName, prms).Tables[0]);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7499"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7500"> 
		///		<ExpectedInput>Existing Guid</ExpectedInput>
		///		<ExpectedOutput>UniqueDatabaseConstraintViolatedException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts a record into the PoolCounter table and returns the newly inserted row (to cut back on table calls)
		/// </summary>
		/// <returns></returns>
		public static DataRow InsertPoolCounter(System.Guid poolCounterGuid, int counterValue, string counterYear)
		{
            //CR3320 - changed to create SqlParameter[] rather than a DataTable before passing to StoredProcedure class
            SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.InsertPoolCounter.poolcounterguid, System.Data.DbType.Guid),
				new SqlParameter(STOREDPROC.InsertPoolCounter.countervalue, System.Data.SqlDbType.Int),
				new SqlParameter(STOREDPROC.InsertPoolCounter.counteryear, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.InsertPoolCounter.divisioncode, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.InsertPoolCounter.recordstatuscode, System.Data.SqlDbType.Char),
  				new SqlParameter(STOREDPROC.InsertPoolCounter.lastupdatefunctionid, System.Data.SqlDbType.Int),
                new SqlParameter(STOREDPROC.InsertPoolCounter.lastupdateuser, System.Data.SqlDbType.VarChar)
			};

            prms[0].Value = poolCounterGuid;
            prms[1].Value = counterValue;
            prms[2].Value = counterYear;
            prms[3].Value = Common.LogonUser.LogonUserDivisionCode;
            prms[4].Value = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
            prms[5].Value = Common.UpdateFunction.UC024FrmPoolUnits;
            prms[6].Value = Common.LogonUser.LogonUserName;     //Normally automatic added in StoredProc execution, but this is a Get that's first inserting

		    return Common.StoredProcedure.GetDataTable(STOREDPROC.InsertPoolCounter.StoredProcName, prms).Rows[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7501"> 
		///		<ExpectedInput>DataTable of PoolCounter record to inactivate</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7502"> 
		///		<ExpectedInput>Invalid DataRow data</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a table schema for spUpdatePoolCounter
		/// </summary>
		/// <returns></returns>
		public static bool InactivatePoolCounter(System.Data.DataRow drPoolCounter)
		{
            //CR3320 - changed input parm from DataTable to DataRow to simplify logic
            SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.UpdatePoolCounter.poolcounterguid, System.Data.DbType.Guid),
				new SqlParameter(STOREDPROC.UpdatePoolCounter.recordstatuscode, System.Data.SqlDbType.Char),
  				new SqlParameter(STOREDPROC.UpdatePoolCounter.lastupdatefunctionid, System.Data.SqlDbType.Int),
                new SqlParameter(STOREDPROC.UpdatePoolCounter.rowversion, System.Data.SqlDbType.Binary),
                new SqlParameter(STOREDPROC.UpdatePoolCounter.lastupdateuser, System.Data.SqlDbType.VarChar)
			};

		    prms[0].Value = drPoolCounter[TABLE.PoolCounter.PoolCounterGuid];
            prms[1].Value = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
            prms[2].Value = Common.UpdateFunction.UC024FrmPoolUnits;
            prms[3].Value = drPoolCounter[TABLE.PoolCounter.RowVersion];
            prms[4].Value = Common.LogonUser.LogonUserName;

            int retValue = new Common.StoredProcedure().TransactionalGetValue(STOREDPROC.UpdatePoolCounter.StoredProcName, prms);

			return(retValue==0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="928"> 
		///		<ExpectedInput>BloodUnitModification Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable of ModificationEquipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="929"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns the equipment associated with a modification
		/// </summary>
		/// <param name="bloodUnitModificationGuid"></param>
		/// <returns></returns>
		public static DataTable GetModificationEquipment(System.Guid bloodUnitModificationGuid)
		{
			SqlParameter[] _prms =
			{
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetModifiedUnitEquipment.bloodunitmodificationguid, bloodUnitModificationGuid)
			};
            			
			return Common.StoredProcedure.GetData(STOREDPROC.GetModifiedUnitEquipment.StoredProcName, _prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="930"> 
		///		<ExpectedInput>BloodUnitModificationGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Supplies for Modification</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="932"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns the supplies associated with a modification
		/// </summary>
		/// <param name="bloodUnitModificationGuid"></param>
		/// <returns></returns>
		public static DataTable GetModificationSupplies(System.Guid bloodUnitModificationGuid)
		{
			SqlParameter[] _prms =
			{
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetModifiedUnitSupplies.bloodunitmodificationguid, bloodUnitModificationGuid)
			};
            			
			return Common.StoredProcedure.GetData(STOREDPROC.GetModifiedUnitSupplies.StoredProcName, _prms).Tables[0];
		}

		

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="956"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of BloodUnits that have been pooled</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="957"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a list of bloodunits that were created via the pooling modification process
		/// </summary>
		/// <returns></returns>
		public static DataTable GetPooledBloodUnits()
		{
			SqlParameter[] _prms =
				{
					Common.StoredProcedure.MakeParameter(STOREDPROC.GetPooledBloodUnits.divisioncode, Common.LogonUser.LogonUserDivisionCode)
				};

			return Common.StoredProcedure.GetData(STOREDPROC.GetPooledBloodUnits.StoredProcName, _prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="958"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of the Units that make up the pooled unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="963"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a list of the units that make up a pooled unit
		/// </summary>
		/// <param name="toBloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitsInPool(System.Guid toBloodUnitGuid)
		{
			SqlParameter[] _prms = 
				{
					Common.StoredProcedure.MakeParameter(STOREDPROC.GetBloodUnitsInPool.tobloodunitguid, toBloodUnitGuid),
					Common.StoredProcedure.MakeParameter(STOREDPROC.GetBloodUnitsInPool.divisioncode, Common.LogonUser.LogonUserDivisionCode)
				};

			return Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitsInPool.StoredProcName, _prms).Tables[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/1/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8385"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8386"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a schema to work with spUpdateBloodUnitTestTable
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyBloodUnitTestTable()
		{
			System.Data.DataTable dtEmpty = new DataTable();

			dtEmpty.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestResultId, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.EntryTechId, typeof(string));
			
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitTest.LastUpdateFunctionId, typeof(int));

			return(dtEmpty);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="964"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Parameter Table for spInsertShipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="966"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates and returns a blank Shipment table to be used in the spInsertShipment sproc
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyShipmentTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.Shipment.TableName);

			dtEmpty.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.Shipment.LocalSupplierGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.Shipment.ProductShipperGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.Shipment.ShipmentDirection, typeof(bool));
			dtEmpty.Columns.Add(TABLE.Shipment.ShipmentDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.Shipment.InvoiceNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.Shipment.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.Shipment.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.Shipment.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.Shipment.RecordStatusCode, typeof(string));

			return dtEmpty;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4275"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Parameter Table for spInsertBloodUnitShipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4276"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates and returns a blank table for adding a bloodunitshipment. Used in spInsertBloodUnitShipment sproc
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyBloodUnitShipmentTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.BloodUnitShipment.TableName);

			dtEmpty.Columns.Add(TABLE.BloodUnitShipment.BloodUnitShipmentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitShipment.BloodUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitShipment.IncomingShipmentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitShipment.LastUpdateUser, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitShipment.RecordStatusCode, typeof(string));

			return dtEmpty;
		}


		///<Developers>
		///	<Developer>Software Architectects Inc.</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7503"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7504"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates and returns a blank table for adding a orderedunit.  Used in spInsertOrderedUnit
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyOrderedUnitTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.OrderedUnit.TableName);

			dtEmpty.Columns.Add(TABLE.OrderedUnit.OrderedUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.OrderedComponentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.OrderedTestGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.BloodUnitGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.SelectedDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.SelectedTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.CrossmatchRequiredIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.CrossmatchDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.CrossmatchResultCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.CrossmatchTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.RecordStatusCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.OrderedUnit.LastUpdateUser, typeof(string));
			return dtEmpty;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="984"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="988"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Datatable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates and returns a blank table for modified bloodunits. Used in spInsertBloodUnitModification sproc
		/// </summary>
		/// <param name="isUpdate">If true, create the spUpdateBloodUnitModification table layout</param>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyBloodUnitModificationTable(bool isUpdate)
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.BloodUnitModification.TableName);

			dtEmpty.Columns.Add(TABLE.BloodUnitModification.BloodUnitModificationGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.ModificationDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.ModificationTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.NewLabelVerificationDatetime, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.DivisionCode, typeof(string));			
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.PoolEditedIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.Comments, typeof(string));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.OriginalUnitIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.BloodUnitModification.ModificationCost, typeof(decimal));

			if (isUpdate)
			{
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.RecordStatusCode, typeof(string));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.RowVersion, typeof(byte[]));	
				
			}
			else
			{
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.FromBloodUnitGuid, typeof(System.Guid));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.ModificationMethodCode, typeof(string));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.NewContainerWeldIndicator, typeof(bool));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.ProductModificationCode, typeof(string));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.ToBloodUnitGuid, typeof(System.Guid));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.FromBloodProductGuid, typeof(System.Guid));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.FromOriginalVolume, typeof(decimal));
				dtEmpty.Columns.Add(TABLE.BloodUnitModification.FromUnitExpirationDate, typeof(System.DateTime));
			}

			return dtEmpty;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1002"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1020"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates a blank table for modified bloodunits's supplies. Used in spInsertModificationSupply sproc
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyModificationSupplyTable(bool isUpdate)
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.ModificationSupply.TableName);

			dtEmpty.Columns.Add(TABLE.ModificationSupply.ModificationSupplyGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.ModificationSupply.RecordStatusCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.ModificationSupply.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.ModificationSupply.BloodUnitModificationGuid, typeof(System.Guid));

			if (isUpdate)
			{
				dtEmpty.Columns.Add(TABLE.ModificationSupply.RowVersion, typeof(byte[]));
			}
			else
			{
				dtEmpty.Columns.Add(TABLE.ModificationSupply.SupplyGuid, typeof(System.Guid));
			}
			

			return dtEmpty;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1025"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Parameter Table for spInsertModificationEquipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1030"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates a blank table for modified bloodunits's equipment. Used in spInsertModificationEquipment sproc
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyModificationEquipmentTable(bool isUpdate)
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(TABLE.ModificationEquipment.TableName);

			dtEmpty.Columns.Add(TABLE.ModificationEquipment.ModificationEquipmentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.ModificationEquipment.RecordStatusCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.ModificationEquipment.LastUpdateFunctionId, typeof(int));
			dtEmpty.Columns.Add(TABLE.ModificationEquipment.BloodUnitModificationGuid, typeof(System.Guid));

			if (isUpdate)
			{
				dtEmpty.Columns.Add(TABLE.ModificationEquipment.RowVersion, typeof(byte[]));
			}
			else
			{	
				dtEmpty.Columns.Add(TABLE.ModificationEquipment.EquipmentGuid, typeof(System.Guid));
			}
			

			return dtEmpty;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1041"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Gets the Original BloodUnits that make up a Modification</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1047"> 
		///		<ExpectedInput>Invalid GUid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>		
		/// <summary>
		/// Static function returns all the active units created by the modification of the suppled blood unit
		/// </summary>
		/// <param name="bloodUnitGuid">Parent blood unit unique identifier</param>
		/// <returns>Datatable of children units</returns>
		public static DataTable GetModifiedBloodUnitChildren(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetModifiedBloodUnitChildren.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetModifiedBloodUnitChildren.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetModifiedBloodUnitChildren.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7273"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7274"> 
		///		<ExpectedInput>Invalid Blood Unit Guid</ExpectedInput>
		///		<ExpectedOutput>Empty blood unit modification table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns all the active blood unit modification records with the suppled blood unit as the parent
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Guid</param>
		/// <returns>Blood Unit Modification table</returns>
		public static DataTable GetBloodUnitModificationsByUnitGuid(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitModificationsByUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitModificationsByUnit.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitModificationsByUnit.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		/// <summary>
		/// Static function returns all the active blood unit modification records with the suppled blood unit as the child
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Guid</param>
		/// <returns>Blood Unit Modification table</returns>
		public static DataTable GetBloodUnitModificationsForChildUnit(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitModificationsByUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitModificationsByUnit.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitModificationsForChildUnit.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5451"> 
		///		<ExpectedInput>Valid dates</ExpectedInput>
		///		<ExpectedOutput>DataTable containing UnitModification data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5452"> 
		///		<ExpectedInput>Invalid dates</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of unit modifications for a date range
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <returns></returns>
		public static DataTable GetUnitModifications(DateTime startDate, DateTime endDate) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetUnitModifications.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetUnitModifications.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetUnitModifications.divisioncode, System.Data.SqlDbType.Char,5)
			};
				
			prms[0].Value = startDate;
			prms[1].Value = endDate;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;
				
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetUnitModifications.StoredProcName, prms);
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/30/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7338"> 
		///		<ExpectedInput>Valid modified unit BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7339"> 
		///		<ExpectedInput>Invalid BloodUnit Guid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the blood unit shipment record for a given unit guid;
		/// used to inactivate a shipment record when a pool/split
		/// unit is inactivated
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns>DataTable</returns>
		public static DataRow GetMostRecentBloodUnitShipmentRecordForBloodUnit(Guid bloodUnitGuid) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetMostRecentBloodUnitShipmentRecordForBloodUnit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = bloodUnitGuid;
			//
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetMostRecentBloodUnitShipmentRecordForBloodUnit.StoredProcName, prms);
			//
			return ds.Tables[0].Rows[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>07/01/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7340"> 
		///		<ExpectedInput>Valid pooled bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7341"> 
		///		<ExpectedInput>BloodUnitGuid that hasn't been split/pooled</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks whether blood unit was pooled or split
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns>True/False</returns>
		public static bool WasBloodUnitPooledOrSplit(Guid bloodUnitGuid) 
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.WasBloodUnitPooledOrSplit.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			//
			prms[0].Value = bloodUnitGuid;
			//
			return new StoredProcedure().GetBoolReturnValue( STOREDPROC.WasBloodUnitPooledOrSplit.StoredProcName, prms );
		}
	}
}
